/** @file   spritetools.cpp
 * @brief   Implementation of SpriteTools class.
 * @version $Revision: 1.2 $
 * @date    $Date: 2006/01/29 20:13:43 $
 * @author  Tomi Lamminsaari
 */


#include "spritetools.h"
#include "filelist.h"
using std::vector;
using std::string;

namespace eng2d {

//********************************************************************
//                                                                   *
//      Static members and constants                                 *
//                                                                   *
//********************************************************************

/** Rotates the given sprite full circle and constructs 'frameCount'
 * number of frames.
 */
vector<BITMAP*> SpriteTools::getRotatedSprites( BITMAP* pSrc, int frameCount )
{
  vector<BITMAP*> bmtable;
  if ( pSrc == 0 ) {
    return bmtable;
  }
  
  float angleStep = 256.0 / frameCount;
  for ( float a = 0; a < 256.0; a += angleStep ) {
    BITMAP* pTmp = create_bitmap( pSrc->w, pSrc->h );
    clear_to_color( pTmp, makecol(255,0,255) );
    fixed angle = ftofix( a );
    
    rotate_sprite( pTmp, pSrc, 0,0, angle );
    bmtable.push_back( pTmp );
  }
  return bmtable;
}



/** As above but the returned vector has pointers to RLE-packed sprites.
 */
vector<RLE_SPRITE*> SpriteTools::getRotatedRLE( BITMAP* pSrc, int frameCount )
{
  vector<RLE_SPRITE*> bmtable;
  if ( pSrc == 0 ) {
    return bmtable;
  }
  
  float angleStep = 256.0 / frameCount;
  BITMAP* pTmp = create_bitmap( pSrc->w, pSrc->h );
  for (float a = 0; a < 256.0; a += angleStep) {
    clear_to_color( pTmp, makecol(255,0,255) );
    fixed angle = ftofix( a );
    
    rotate_sprite( pTmp, pSrc, 0,0, angle );
    RLE_SPRITE* pRLE = get_rle_sprite( pTmp );
    bmtable.push_back( pRLE );
  }
  destroy_bitmap( pTmp );
  return bmtable;
}


/** Destroys all the bitmaps in given vector.
 */
void SpriteTools::flushBitmaps( vector<BITMAP*>& bmtable )
{
  for (int i=0; i < bmtable.size(); i++) {
    if ( bmtable.at(i) != 0 ) {
      destroy_bitmap( bmtable.at(i) );
      bmtable.at(i) = 0;
    }
  }
}



/** Destroys the rle-sprites from given vector.
 */
void SpriteTools::flushRLE( vector<RLE_SPRITE*>& bmtable )
{
  for (int i=0; i < bmtable.size(); i++) {
    if ( bmtable.at(i) != 0) {
      destroy_rle_sprite( bmtable.at(i) );
      bmtable.at(i) = 0;
    }
  }
}



/** Loads the bitmap from given file and converts it to rle-sprite.
 */
RLE_SPRITE* SpriteTools::load_RLE_Sprite( const string& filename )
{
  BITMAP* pTmp = load_bitmap( filename.c_str(), 0 );
  if ( pTmp == 0 ) {
    return 0;
  }

  RLE_SPRITE* pRLE = get_rle_sprite( pTmp );
  destroy_bitmap( pTmp );
  return pRLE;
}



/** Loads a set of bitmaps.
 */
int SpriteTools::loadBitmapSet( const string& filename, vector<BITMAP*>* pT )
{
  FileList flist( filename );
  if ( flist.bad() == true ) {
    return -1;
  }
  
  // Try to load the bitmapfiles listed in FileList.
  int ret = 0;
  for ( int i=0; i < flist.filecount(); i++ ) {
    string fname = flist.getFile( i );
    BITMAP* pB = load_bitmap( fname.c_str(), 0 );
    pT->push_back( pB );
    if ( pB == 0 ) {
      ret = -1;
    }
  }
  return ret;
}



/** Loads a set of rle-sprites.
 */
int SpriteTools::loadRleSpriteSet( const string& filename, vector<RLE_SPRITE*>* pT )
{
  FileList flist( filename );
  if ( flist.bad() == true ) {
    return -1;
  }
  
  // Try to load the bitmaps and convert them to rle-sprites.
  int ret = 0;
  for ( int i=0; i < flist.filecount(); i++ ) {
    string fname = flist.getFile( i );
    BITMAP* pB = load_bitmap( fname.c_str(), 0 );
    if ( pB != 0 ) {
      pT->push_back( get_rle_sprite(pB) );
      destroy_bitmap( pB );
      pB = 0;
      
    } else {
      pT->push_back( 0 );
      ret = 0;
      
    }
  }
  return ret;
}


/** Converts the bitmaps from format to another.
 */
void* SpriteTools::ConvertToGraphicFormat( const BITMAP* aSource,
                                           GfxContentType aConvertTo )
{
  void* gfx = 0;
  switch ( aConvertTo ) {
    case ( ESprite ):
    case ( EBitmap ): {
      BITMAP* clone = create_bitmap( aSource->w, aSource->h );
      if ( clone == 0 ) {
        return 0;
      }
      blit( const_cast<BITMAP*>(aSource), clone, 0,0, 0,0, aSource->w, aSource->h );
      gfx = clone;
      break;
    }
    case ( ERleSprite ): {
      RLE_SPRITE* clone = get_rle_sprite( const_cast<BITMAP*>(aSource) );
      gfx = clone;
      break;
    }
    default: {
      gfx = 0;
      break;
    }
  }
  return gfx;   
}

}
